@extends('layouts.app')

@section('content')
<div class="card">
    <div class="card-header">
        <h3 class="card-title"><i class="fas fa-book"></i> New Journal Voucher</h3>
        <a href="{{ route('vouchers.journal.index') }}" class="btn btn-outline">
            <i class="fas fa-arrow-left"></i> Back to List
        </a>
    </div>
    <div class="card-body">
        @if($errors->any())
        <div class="alert alert-danger">
            <ul style="margin: 0; padding-left: 1rem;">
                @foreach($errors->all() as $error)
                <li>{{ $error }}</li>
                @endforeach
            </ul>
        </div>
        @endif

        <form action="{{ route('vouchers.journal.store') }}" method="POST" id="journalForm">
            @csrf
            
            <div class="form-row">
                <div class="form-group" style="flex: 0 0 180px;">
                    <label class="form-label">Voucher No. <span class="text-danger">*</span></label>
                    <input type="text" name="voucher_no" class="form-control" 
                           value="{{ old('voucher_no', $nextVoucherNo) }}" readonly 
                           style="background: #f0f0f0; font-weight: bold;">
                </div>
                <div class="form-group" style="flex: 0 0 180px;">
                    <label class="form-label">Date <span class="text-danger">*</span></label>
                    <input type="date" name="date" class="form-control" 
                           value="{{ old('date', date('Y-m-d')) }}" required>
                </div>
            </div>

            <div class="form-group">
                <label class="form-label">Narration</label>
                <textarea name="narration" class="form-control" rows="2" 
                          placeholder="Description of journal entry...">{{ old('narration') }}</textarea>
            </div>

            <!-- Journal Entries -->
            <div class="card mt-4" style="border: 1px solid var(--border-color);">
                <div class="card-header" style="background: var(--primary-color); color: #fff; padding: 0.75rem 1rem;">
                    <h4 style="font-size: 0.9375rem; font-weight: 600; margin: 0;">
                        <i class="fas fa-list"></i> Journal Entries
                    </h4>
                    <button type="button" class="btn btn-outline" onclick="addEntry()" 
                            style="padding: 0.25rem 0.75rem; background: #fff; color: var(--primary-color);">
                        <i class="fas fa-plus"></i> Add Row
                    </button>
                </div>
                <div class="card-body" style="padding: 0;">
                    <table id="entriesTable">
                        <thead>
                            <tr>
                                <th>Account <span class="text-danger">*</span></th>
                                <th class="text-right" style="width: 150px;">Debit</th>
                                <th class="text-right" style="width: 150px;">Credit</th>
                                <th style="width: 60px;">Action</th>
                            </tr>
                        </thead>
                        <tbody id="entriesBody">
                            <!-- Entries will be added here -->
                        </tbody>
                        <tfoot>
                            <tr style="background: var(--light-bg);">
                                <td class="text-right"><strong>Totals:</strong></td>
                                <td class="text-right">
                                    <strong id="totalDebit" class="text-success">0.00</strong>
                                </td>
                                <td class="text-right">
                                    <strong id="totalCredit" class="text-danger">0.00</strong>
                                </td>
                                <td></td>
                            </tr>
                            <tr id="differenceRow" style="display: none; background: #ffcccc;">
                                <td class="text-right"><strong>Difference:</strong></td>
                                <td colspan="2" class="text-center">
                                    <strong id="difference" class="text-danger">0.00</strong>
                                </td>
                                <td></td>
                            </tr>
                        </tfoot>
                    </table>
                </div>
            </div>

            <div id="balanceAlert" class="alert alert-danger mt-3" style="display: none;">
                <i class="fas fa-exclamation-triangle"></i> 
                <strong>Debit and Credit must be equal!</strong> Please adjust entries before saving.
            </div>

            <div id="emptyAlert" class="alert alert-warning mt-3" style="display: none;">
                <i class="fas fa-info-circle"></i> 
                <strong>Please add at least 2 entries</strong> - one debit and one credit entry.
            </div>

            <div style="display: flex; gap: 0.5rem; margin-top: 1.5rem;">
                <button type="submit" class="btn btn-primary" id="submitBtn" disabled>
                    <i class="fas fa-save"></i> Save Journal Voucher
                </button>
                <a href="{{ route('vouchers.journal.index') }}" class="btn btn-outline">
                    <i class="fas fa-times"></i> Cancel
                </a>
            </div>
        </form>

        <!-- Info Box -->
        <div class="alert alert-info mt-4">
            <h4 style="font-size: 0.9375rem; margin-bottom: 0.5rem;">
                <i class="fas fa-info-circle"></i> Journal Voucher
            </h4>
            <p style="margin: 0; font-size: 0.875rem;">
                <strong>Use journal vouchers for:</strong> Adjustments, corrections, depreciation, 
                provisions, accruals, and any entry where multiple accounts are involved. 
                Total debits MUST equal total credits.
            </p>
        </div>
    </div>
</div>

<script>
const accounts = @json($accounts);
let entryCount = 0;

function createAccountSelect(name, index) {
    let html = `<select name="${name}" class="form-control" required>`;
    html += '<option value="">-- Select Account --</option>';
    accounts.forEach(account => {
        html += `<option value="${account.id}">${account.code} - ${account.name}</option>`;
    });
    html += '</select>';
    return html;
}

function addEntry() {
    const tbody = document.getElementById('entriesBody');
    const row = document.createElement('tr');
    row.id = `entry_${entryCount}`;
    row.innerHTML = `
        <td>${createAccountSelect('entries[' + entryCount + '][account_id]', entryCount)}</td>
        <td>
            <input type="number" name="entries[${entryCount}][debit]" class="form-control text-right debit-input" 
                   value="0" min="0" step="0.01" onchange="updateTotals()" onkeyup="updateTotals()">
        </td>
        <td>
            <input type="number" name="entries[${entryCount}][credit]" class="form-control text-right credit-input" 
                   value="0" min="0" step="0.01" onchange="updateTotals()" onkeyup="updateTotals()">
        </td>
        <td class="text-center">
            <button type="button" class="btn btn-outline" onclick="removeEntry(${entryCount})" 
                    style="padding: 0.25rem 0.5rem; color: var(--danger-color);">
                <i class="fas fa-trash"></i>
            </button>
        </td>
    `;
    tbody.appendChild(row);
    entryCount++;
    updateTotals();
}

function removeEntry(index) {
    const row = document.getElementById(`entry_${index}`);
    if (row) {
        row.remove();
        updateTotals();
    }
}

function updateTotals() {
    let totalDebit = 0;
    let totalCredit = 0;
    
    document.querySelectorAll('.debit-input').forEach(input => {
        totalDebit += parseFloat(input.value) || 0;
    });
    
    document.querySelectorAll('.credit-input').forEach(input => {
        totalCredit += parseFloat(input.value) || 0;
    });
    
    document.getElementById('totalDebit').textContent = totalDebit.toFixed(2);
    document.getElementById('totalCredit').textContent = totalCredit.toFixed(2);
    
    const difference = Math.abs(totalDebit - totalCredit);
    document.getElementById('difference').textContent = difference.toFixed(2);
    
    const entriesCount = document.querySelectorAll('#entriesBody tr').length;
    const isBalanced = difference < 0.01 && totalDebit > 0;
    const hasEntries = entriesCount >= 2;
    
    document.getElementById('differenceRow').style.display = difference >= 0.01 ? 'table-row' : 'none';
    document.getElementById('balanceAlert').style.display = difference >= 0.01 && totalDebit > 0 ? 'block' : 'none';
    document.getElementById('emptyAlert').style.display = entriesCount < 2 ? 'block' : 'none';
    document.getElementById('submitBtn').disabled = !(isBalanced && hasEntries);
}

// Initialize with 2 empty rows
document.addEventListener('DOMContentLoaded', function() {
    addEntry();
    addEntry();
});
</script>
@endsection
