<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphMany;

class Purchase extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'bill_no',
        'date',
        'party_id',
        'contract_id',
        'account_id',
        'payment_type',
        'sub_total',
        'discount_percent',
        'discount_amount',
        'tax_percent',
        'tax_amount',
        'other_charges',
        'total_amount',
        'paid_amount',
        'balance_amount',
        'narration',
        'status',
        'created_by',
    ];

    protected $casts = [
        'date' => 'date',
        'sub_total' => 'decimal:2',
        'discount_percent' => 'decimal:2',
        'discount_amount' => 'decimal:2',
        'tax_percent' => 'decimal:2',
        'tax_amount' => 'decimal:2',
        'other_charges' => 'decimal:2',
        'total_amount' => 'decimal:2',
        'paid_amount' => 'decimal:2',
        'balance_amount' => 'decimal:2',
    ];

    /**
     * Party (supplier) relationship
     */
    public function party(): BelongsTo
    {
        return $this->belongsTo(Party::class);
    }

    /**
     * Contract relationship
     */
    public function contract(): BelongsTo
    {
        return $this->belongsTo(Contract::class);
    }

    /**
     * Payment account relationship
     */
    public function account(): BelongsTo
    {
        return $this->belongsTo(Account::class);
    }

    /**
     * Created by user relationship
     */
    public function createdBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Purchase items relationship
     */
    public function items(): HasMany
    {
        return $this->hasMany(PurchaseItem::class);
    }

    /**
     * Transactions relationship
     */
    public function transactions(): MorphMany
    {
        return $this->morphMany(Transaction::class, 'transactionable');
    }

    /**
     * Calculate totals from items
     */
    public function calculateTotals(): void
    {
        $this->sub_total = $this->items->sum('amount');
        $this->discount_amount = $this->sub_total * ($this->discount_percent / 100);
        $this->tax_amount = ($this->sub_total - $this->discount_amount) * ($this->tax_percent / 100);
        $this->total_amount = $this->sub_total - $this->discount_amount + $this->tax_amount + $this->other_charges;
        $this->balance_amount = $this->total_amount - $this->paid_amount;
    }

    /**
     * Get payment status
     */
    public function getPaymentStatusAttribute(): string
    {
        if ($this->balance_amount <= 0) {
            return 'paid';
        } elseif ($this->paid_amount > 0) {
            return 'partial';
        }
        return 'unpaid';
    }

    /**
     * Scope for confirmed purchases
     */
    public function scopeConfirmed($query)
    {
        return $query->where('status', 'confirmed');
    }

    /**
     * Scope for date range
     */
    public function scopeDateRange($query, $from, $to)
    {
        return $query->whereBetween('date', [$from, $to]);
    }

    /**
     * Generate next bill number
     */
    public static function generateBillNo(): string
    {
        $prefix = 'PUR-';
        $year = date('y');
        $month = date('m');
        
        $lastPurchase = self::where('bill_no', 'like', $prefix . $year . $month . '%')
            ->orderBy('bill_no', 'desc')
            ->first();

        if ($lastPurchase) {
            $lastNumber = intval(substr($lastPurchase->bill_no, -4));
            return $prefix . $year . $month . str_pad($lastNumber + 1, 4, '0', STR_PAD_LEFT);
        }

        return $prefix . $year . $month . '0001';
    }

    /**
     * Post to ledger (create transactions)
     */
    public function postToLedger(): void
    {
        // Delete existing transactions
        $this->transactions()->delete();

        if ($this->status !== 'confirmed') {
            return;
        }

        $voucherNo = $this->bill_no;
        $date = $this->date;
        $userId = $this->created_by;

        // 1. Debit: Purchase Account (Expense increases)
        $purchaseAccount = Account::where('sub_type', 'purchase')->first();
        if ($purchaseAccount) {
            $this->transactions()->create([
                'date' => $date,
                'account_id' => $purchaseAccount->id,
                'party_id' => $this->party_id,
                'voucher_type' => 'PUR',
                'voucher_no' => $voucherNo,
                'description' => 'Purchase from ' . $this->party->name,
                'debit' => $this->total_amount,
                'credit' => 0,
                'created_by' => $userId,
            ]);
        }

        // 2. Credit: Party Account (Liability increases) OR Cash/Bank (Asset decreases)
        if ($this->payment_type === 'cash' && $this->account_id) {
            // Cash purchase - Credit Cash/Bank
            $this->transactions()->create([
                'date' => $date,
                'account_id' => $this->account_id,
                'party_id' => $this->party_id,
                'voucher_type' => 'PUR',
                'voucher_no' => $voucherNo,
                'description' => 'Cash purchase from ' . $this->party->name,
                'debit' => 0,
                'credit' => $this->total_amount,
                'created_by' => $userId,
            ]);
        } else {
            // Credit purchase - Credit Party's Account
            if ($this->party->account_id) {
                $this->transactions()->create([
                    'date' => $date,
                    'account_id' => $this->party->account_id,
                    'party_id' => $this->party_id,
                    'voucher_type' => 'PUR',
                    'voucher_no' => $voucherNo,
                    'description' => 'Credit purchase from ' . $this->party->name,
                    'debit' => 0,
                    'credit' => $this->total_amount,
                    'created_by' => $userId,
                ]);
            }

            // If partial payment, also record cash payment
            if ($this->paid_amount > 0 && $this->account_id) {
                $this->transactions()->create([
                    'date' => $date,
                    'account_id' => $this->party->account_id,
                    'party_id' => $this->party_id,
                    'voucher_type' => 'PUR',
                    'voucher_no' => $voucherNo,
                    'description' => 'Payment against purchase',
                    'debit' => $this->paid_amount,
                    'credit' => 0,
                    'created_by' => $userId,
                ]);

                $this->transactions()->create([
                    'date' => $date,
                    'account_id' => $this->account_id,
                    'party_id' => $this->party_id,
                    'voucher_type' => 'PUR',
                    'voucher_no' => $voucherNo,
                    'description' => 'Payment for purchase',
                    'debit' => 0,
                    'credit' => $this->paid_amount,
                    'created_by' => $userId,
                ]);
            }
        }

        // Update account balances
        $this->updateAccountBalances();
    }

    /**
     * Update all related account balances
     */
    protected function updateAccountBalances(): void
    {
        foreach ($this->transactions as $transaction) {
            $transaction->account->updateBalance();
        }

        $this->party->updateBalance();
    }

    /**
     * Update item stocks
     */
    public function updateStock(): void
    {
        foreach ($this->items as $purchaseItem) {
            $purchaseItem->item->addStock($purchaseItem->quantity);
        }
    }
}
