<?php

namespace App\Http\Controllers;

use App\Models\Account;
use App\Models\Item;
use App\Models\Party;
use App\Models\Purchase;
use App\Models\Sale;
use App\Models\Transaction;
use App\Models\Voucher;
use App\Models\VoucherEntry;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\View\View;

class ReportController extends Controller
{
    /**
     * Daily Book Report - All transactions for a specific date
     */
    public function dailyBook(Request $request): View
    {
        $date = $request->get('date', now()->format('Y-m-d'));
        
        $transactions = Transaction::with(['account', 'transactionable'])
            ->whereDate('date', $date)
            ->orderBy('created_at')
            ->get();
        
        $summary = [
            'total_debit' => $transactions->sum('debit'),
            'total_credit' => $transactions->sum('credit'),
        ];
        
        return view('reports.daily-book', compact('transactions', 'date', 'summary'));
    }

    /**
     * Journal Register Report - All voucher entries
     */
    public function journal(Request $request): View
    {
        $fromDate = $request->get('from_date', now()->startOfMonth()->format('Y-m-d'));
        $toDate = $request->get('to_date', now()->format('Y-m-d'));
        
        $vouchers = Voucher::with(['entries.account', 'user'])
            ->whereBetween('date', [$fromDate, $toDate])
            ->orderBy('date')
            ->orderBy('voucher_no')
            ->get();
        
        return view('reports.journal', compact('vouchers', 'fromDate', 'toDate'));
    }

    /**
     * A/C Ledger Report - Individual account ledger
     */
    public function ledger(Request $request): View
    {
        $accounts = Account::orderBy('code')->get();
        $selectedAccount = $request->get('account_id');
        $fromDate = $request->get('from_date', now()->startOfMonth()->format('Y-m-d'));
        $toDate = $request->get('to_date', now()->format('Y-m-d'));
        
        $entries = collect([]);
        $openingBalance = 0;
        $account = null;
        
        if ($selectedAccount) {
            $account = Account::find($selectedAccount);
            
            // Calculate opening balance (before from_date)
            $openingBalance = Transaction::where('account_id', $selectedAccount)
                ->whereDate('date', '<', $fromDate)
                ->selectRaw('COALESCE(SUM(debit), 0) - COALESCE(SUM(credit), 0) as balance')
                ->value('balance') ?? 0;
            
            // Get transactions within date range
            $entries = Transaction::with('transactionable')
                ->where('account_id', $selectedAccount)
                ->whereBetween('date', [$fromDate, $toDate])
                ->orderBy('date')
                ->orderBy('created_at')
                ->get();
        }
        
        return view('reports.ledger', compact('accounts', 'selectedAccount', 'account', 'entries', 'fromDate', 'toDate', 'openingBalance'));
    }

    /**
     * A/C Ledger Code Range Report
     */
    public function ledgerCodeRange(Request $request): View
    {
        $fromCode = $request->get('from_code');
        $toCode = $request->get('to_code');
        $fromDate = $request->get('from_date', now()->startOfMonth()->format('Y-m-d'));
        $toDate = $request->get('to_date', now()->format('Y-m-d'));
        
        $accounts = collect([]);
        
        if ($fromCode && $toCode) {
            $accounts = Account::whereBetween('code', [$fromCode, $toCode])
                ->orderBy('code')
                ->get()
                ->map(function ($account) use ($fromDate, $toDate) {
                    $account->opening = Transaction::where('account_id', $account->id)
                        ->whereDate('date', '<', $fromDate)
                        ->selectRaw('COALESCE(SUM(debit), 0) - COALESCE(SUM(credit), 0) as balance')
                        ->value('balance') ?? 0;
                    
                    $period = Transaction::where('account_id', $account->id)
                        ->whereBetween('date', [$fromDate, $toDate])
                        ->selectRaw('COALESCE(SUM(debit), 0) as debit, COALESCE(SUM(credit), 0) as credit')
                        ->first();
                    
                    $account->period_debit = $period->debit ?? 0;
                    $account->period_credit = $period->credit ?? 0;
                    $account->closing = $account->opening + $account->period_debit - $account->period_credit;
                    
                    return $account;
                });
        }
        
        return view('reports.ledger-code-range', compact('accounts', 'fromCode', 'toCode', 'fromDate', 'toDate'));
    }

    /**
     * Trial Balance Report
     */
    public function trialBalance(Request $request): View
    {
        $asOnDate = $request->get('as_on_date', now()->format('Y-m-d'));
        
        $accounts = Account::orderBy('code')
            ->get()
            ->map(function ($account) use ($asOnDate) {
                $balance = Transaction::where('account_id', $account->id)
                    ->whereDate('date', '<=', $asOnDate)
                    ->selectRaw('COALESCE(SUM(debit), 0) - COALESCE(SUM(credit), 0) as balance')
                    ->value('balance') ?? 0;
                
                $account->debit_balance = $balance > 0 ? $balance : 0;
                $account->credit_balance = $balance < 0 ? abs($balance) : 0;
                
                return $account;
            })
            ->filter(function ($account) {
                return $account->debit_balance != 0 || $account->credit_balance != 0;
            });
        
        $totalDebit = $accounts->sum('debit_balance');
        $totalCredit = $accounts->sum('credit_balance');
        
        return view('reports.trial-balance', compact('accounts', 'asOnDate', 'totalDebit', 'totalCredit'));
    }

    /**
     * Trial Balance Specific Head Report
     */
    public function trialBalanceSpecific(Request $request): View
    {
        $type = $request->get('type');
        $asOnDate = $request->get('as_on_date', now()->format('Y-m-d'));
        $types = Account::TYPES;
        
        $accounts = collect([]);
        
        if ($type) {
            $accounts = Account::where('type', $type)
                ->orderBy('code')
                ->get()
                ->map(function ($account) use ($asOnDate) {
                    $balance = Transaction::where('account_id', $account->id)
                        ->whereDate('date', '<=', $asOnDate)
                        ->selectRaw('COALESCE(SUM(debit), 0) - COALESCE(SUM(credit), 0) as balance')
                        ->value('balance') ?? 0;
                    
                    $account->balance = $balance;
                    return $account;
                });
        }
        
        return view('reports.trial-balance-specific', compact('accounts', 'type', 'types', 'asOnDate'));
    }

    /**
     * Cash Book Report
     */
    public function cashBook(Request $request): View
    {
        $fromDate = $request->get('from_date', now()->startOfMonth()->format('Y-m-d'));
        $toDate = $request->get('to_date', now()->format('Y-m-d'));
        
        // Get Cash account
        $cashAccount = Account::where('sub_type', 'Cash in Hand')->first();
        
        $entries = collect([]);
        $openingBalance = 0;
        
        if ($cashAccount) {
            $openingBalance = Transaction::where('account_id', $cashAccount->id)
                ->whereDate('date', '<', $fromDate)
                ->selectRaw('COALESCE(SUM(debit), 0) - COALESCE(SUM(credit), 0) as balance')
                ->value('balance') ?? 0;
            
            $entries = Transaction::with('transactionable')
                ->where('account_id', $cashAccount->id)
                ->whereBetween('date', [$fromDate, $toDate])
                ->orderBy('date')
                ->orderBy('created_at')
                ->get();
        }
        
        $closingBalance = $openingBalance + $entries->sum('debit') - $entries->sum('credit');
        
        return view('reports.cash-book', compact('entries', 'fromDate', 'toDate', 'openingBalance', 'closingBalance', 'cashAccount'));
    }

    /**
     * Cash Flow Statement Report
     */
    public function cashFlow(Request $request): View
    {
        $fromDate = $request->get('from_date', now()->startOfMonth()->format('Y-m-d'));
        $toDate = $request->get('to_date', now()->format('Y-m-d'));
        
        // Cash & Bank accounts
        $cashBankAccounts = Account::whereIn('sub_type', ['Cash in Hand', 'Bank Accounts'])->pluck('id');
        
        $inflows = Transaction::with('account')
            ->whereIn('account_id', $cashBankAccounts)
            ->whereBetween('date', [$fromDate, $toDate])
            ->where('debit', '>', 0)
            ->selectRaw('account_id, SUM(debit) as amount')
            ->groupBy('account_id')
            ->get();
        
        $outflows = Transaction::with('account')
            ->whereIn('account_id', $cashBankAccounts)
            ->whereBetween('date', [$fromDate, $toDate])
            ->where('credit', '>', 0)
            ->selectRaw('account_id, SUM(credit) as amount')
            ->groupBy('account_id')
            ->get();
        
        $totalInflow = $inflows->sum('amount');
        $totalOutflow = $outflows->sum('amount');
        
        return view('reports.cash-flow', compact('inflows', 'outflows', 'fromDate', 'toDate', 'totalInflow', 'totalOutflow'));
    }

    /**
     * Profit & Loss Report
     */
    public function profitLoss(Request $request): View
    {
        $fromDate = $request->get('from_date', now()->startOfYear()->format('Y-m-d'));
        $toDate = $request->get('to_date', now()->format('Y-m-d'));
        
        // Income accounts (credit balance = income)
        $incomeAccounts = Account::where('type', 'Income')
            ->get()
            ->map(function ($account) use ($fromDate, $toDate) {
                $amount = Transaction::where('account_id', $account->id)
                    ->whereBetween('date', [$fromDate, $toDate])
                    ->selectRaw('COALESCE(SUM(credit), 0) - COALESCE(SUM(debit), 0) as amount')
                    ->value('amount') ?? 0;
                $account->amount = $amount;
                return $account;
            })
            ->filter(fn($a) => $a->amount != 0);
        
        // Expense accounts (debit balance = expense)
        $expenseAccounts = Account::where('type', 'Expense')
            ->get()
            ->map(function ($account) use ($fromDate, $toDate) {
                $amount = Transaction::where('account_id', $account->id)
                    ->whereBetween('date', [$fromDate, $toDate])
                    ->selectRaw('COALESCE(SUM(debit), 0) - COALESCE(SUM(credit), 0) as amount')
                    ->value('amount') ?? 0;
                $account->amount = $amount;
                return $account;
            })
            ->filter(fn($a) => $a->amount != 0);
        
        $totalIncome = $incomeAccounts->sum('amount');
        $totalExpenses = $expenseAccounts->sum('amount');
        $netProfit = $totalIncome - $totalExpenses;
        
        return view('reports.profit-loss', compact('incomeAccounts', 'expenseAccounts', 'fromDate', 'toDate', 'totalIncome', 'totalExpenses', 'netProfit'));
    }

    /**
     * Balance Sheet Report
     */
    public function balanceSheet(Request $request): View
    {
        $asOnDate = $request->get('as_on_date', now()->format('Y-m-d'));
        
        // Assets
        $assets = Account::where('type', 'Asset')
            ->get()
            ->map(function ($account) use ($asOnDate) {
                $balance = Transaction::where('account_id', $account->id)
                    ->whereDate('date', '<=', $asOnDate)
                    ->selectRaw('COALESCE(SUM(debit), 0) - COALESCE(SUM(credit), 0) as balance')
                    ->value('balance') ?? 0;
                $account->balance = $balance;
                return $account;
            })
            ->filter(fn($a) => $a->balance != 0);
        
        // Liabilities
        $liabilities = Account::where('type', 'Liability')
            ->get()
            ->map(function ($account) use ($asOnDate) {
                $balance = Transaction::where('account_id', $account->id)
                    ->whereDate('date', '<=', $asOnDate)
                    ->selectRaw('COALESCE(SUM(credit), 0) - COALESCE(SUM(debit), 0) as balance')
                    ->value('balance') ?? 0;
                $account->balance = $balance;
                return $account;
            })
            ->filter(fn($a) => $a->balance != 0);
        
        // Equity
        $equity = Account::where('type', 'Equity')
            ->get()
            ->map(function ($account) use ($asOnDate) {
                $balance = Transaction::where('account_id', $account->id)
                    ->whereDate('date', '<=', $asOnDate)
                    ->selectRaw('COALESCE(SUM(credit), 0) - COALESCE(SUM(debit), 0) as balance')
                    ->value('balance') ?? 0;
                $account->balance = $balance;
                return $account;
            })
            ->filter(fn($a) => $a->balance != 0);
        
        $totalAssets = $assets->sum('balance');
        $totalLiabilities = $liabilities->sum('balance');
        $totalEquity = $equity->sum('balance');
        
        return view('reports.balance-sheet', compact('assets', 'liabilities', 'equity', 'asOnDate', 'totalAssets', 'totalLiabilities', 'totalEquity'));
    }

    /**
     * Stock Sheet Report
     */
    public function stock(Request $request): View
    {
        $type = $request->get('type');
        
        $query = Item::query();
        
        if ($type) {
            $query->where('type', $type);
        }
        
        $items = $query->orderBy('name')->get();
        
        $totalValue = $items->sum(fn($item) => $item->stock_qty * $item->purchase_price);
        $totalStock = $items->sum('stock_qty');
        
        return view('reports.stock', compact('items', 'type', 'totalValue', 'totalStock'));
    }

    /**
     * Party Balance Report
     */
    public function partyBalance(Request $request): View
    {
        $type = $request->get('type', 'all');
        
        $query = Party::with('account');
        
        if ($type === 'supplier') {
            $query->suppliers();
        } elseif ($type === 'customer') {
            $query->customers();
        }
        
        $parties = $query->get()->map(function ($party) {
            if ($party->account) {
                $balance = Transaction::where('account_id', $party->account->id)
                    ->selectRaw('COALESCE(SUM(debit), 0) - COALESCE(SUM(credit), 0) as balance')
                    ->value('balance') ?? 0;
                $party->balance = $balance;
            } else {
                $party->balance = 0;
            }
            return $party;
        })->filter(fn($p) => $p->balance != 0);
        
        $totalReceivable = $parties->where('balance', '>', 0)->sum('balance');
        $totalPayable = abs($parties->where('balance', '<', 0)->sum('balance'));
        
        return view('reports.party-balance', compact('parties', 'type', 'totalReceivable', 'totalPayable'));
    }

    /**
     * Bill Wise Purchase Report
     */
    public function billWisePurchase(Request $request): View
    {
        $fromDate = $request->get('from_date', now()->startOfMonth()->format('Y-m-d'));
        $toDate = $request->get('to_date', now()->format('Y-m-d'));
        $partyId = $request->get('party_id');
        
        $parties = Party::suppliers()->orderBy('name')->get();
        
        $query = Purchase::with(['party', 'items.item'])
            ->whereBetween('date', [$fromDate, $toDate]);
        
        if ($partyId) {
            $query->where('party_id', $partyId);
        }
        
        $purchases = $query->orderBy('date')->orderBy('bill_no')->get();
        
        $summary = [
            'total_amount' => $purchases->sum('total_amount'),
            'total_paid' => $purchases->sum('paid_amount'),
            'total_due' => $purchases->sum('due_amount'),
            'count' => $purchases->count(),
        ];
        
        return view('reports.bill-wise-purchase', compact('purchases', 'parties', 'fromDate', 'toDate', 'partyId', 'summary'));
    }

    /**
     * Bill Wise Sale Report
     */
    public function billWiseSale(Request $request): View
    {
        $fromDate = $request->get('from_date', now()->startOfMonth()->format('Y-m-d'));
        $toDate = $request->get('to_date', now()->format('Y-m-d'));
        $partyId = $request->get('party_id');
        
        $parties = Party::customers()->orderBy('name')->get();
        
        $query = Sale::with(['party', 'items.item'])
            ->whereBetween('date', [$fromDate, $toDate]);
        
        if ($partyId) {
            $query->where('party_id', $partyId);
        }
        
        $sales = $query->orderBy('date')->orderBy('bill_no')->get();
        
        $summary = [
            'total_amount' => $sales->sum('total_amount'),
            'total_received' => $sales->sum('paid_amount'),
            'total_due' => $sales->sum('due_amount'),
            'count' => $sales->count(),
        ];
        
        return view('reports.bill-wise-sale', compact('sales', 'parties', 'fromDate', 'toDate', 'partyId', 'summary'));
    }

    /**
     * Aging Report - Outstanding receivables/payables
     */
    public function aging(Request $request): View
    {
        $type = $request->get('type', 'receivable');
        $asOnDate = $request->get('as_on_date', now()->format('Y-m-d'));
        
        if ($type === 'receivable') {
            // Outstanding sales
            $entries = Sale::with('party')
                ->where('due_amount', '>', 0)
                ->whereDate('date', '<=', $asOnDate)
                ->get()
                ->map(function ($sale) use ($asOnDate) {
                    $days = now()->parse($asOnDate)->diffInDays($sale->date);
                    $sale->days_outstanding = $days;
                    $sale->aging_bucket = $this->getAgingBucket($days);
                    return $sale;
                });
        } else {
            // Outstanding purchases
            $entries = Purchase::with('party')
                ->where('due_amount', '>', 0)
                ->whereDate('date', '<=', $asOnDate)
                ->get()
                ->map(function ($purchase) use ($asOnDate) {
                    $days = now()->parse($asOnDate)->diffInDays($purchase->date);
                    $purchase->days_outstanding = $days;
                    $purchase->aging_bucket = $this->getAgingBucket($days);
                    return $purchase;
                });
        }
        
        return view('reports.aging', compact('entries', 'type', 'asOnDate'));
    }
    
    private function getAgingBucket($days): string
    {
        if ($days <= 30) return '0-30 Days';
        if ($days <= 60) return '31-60 Days';
        if ($days <= 90) return '61-90 Days';
        return '90+ Days';
    }

    /**
     * Chart of Accounts Report
     */
    public function chartOfAccounts(): View
    {
        $accounts = Account::orderBy('type')
            ->orderBy('sub_type')
            ->orderBy('code')
            ->get()
            ->groupBy('type');
        
        return view('reports.chart-of-accounts', compact('accounts'));
    }

    /**
     * General Ledger Report - All accounts with transactions
     */
    public function generalLedger(Request $request): View
    {
        $fromDate = $request->get('from_date', now()->startOfMonth()->format('Y-m-d'));
        $toDate = $request->get('to_date', now()->format('Y-m-d'));
        
        $accounts = Account::has('transactions')
            ->orderBy('code')
            ->get()
            ->map(function ($account) use ($fromDate, $toDate) {
                $account->opening = Transaction::where('account_id', $account->id)
                    ->whereDate('date', '<', $fromDate)
                    ->selectRaw('COALESCE(SUM(debit), 0) - COALESCE(SUM(credit), 0) as balance')
                    ->value('balance') ?? 0;
                
                $account->transactions = Transaction::with('transactionable')
                    ->where('account_id', $account->id)
                    ->whereBetween('date', [$fromDate, $toDate])
                    ->orderBy('date')
                    ->get();
                
                $account->closing = $account->opening + 
                    $account->transactions->sum('debit') - 
                    $account->transactions->sum('credit');
                
                return $account;
            })
            ->filter(fn($a) => $a->transactions->count() > 0 || $a->opening != 0);
        
        return view('reports.general-ledger', compact('accounts', 'fromDate', 'toDate'));
    }
}
