<?php

namespace App\Http\Controllers;

use App\Models\Item;
use App\Models\Production;
use App\Models\ProductionItem;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\View\View;

class ProductionController extends Controller
{
    /**
     * Display a listing of productions.
     */
    public function index(Request $request): View
    {
        $query = Production::with(['inputItems.item', 'outputItems.item', 'creator']);

        // Date filter
        if ($request->filled('from_date')) {
            $query->whereDate('date', '>=', $request->from_date);
        }
        if ($request->filled('to_date')) {
            $query->whereDate('date', '<=', $request->to_date);
        }

        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('batch_no', 'like', "%{$search}%")
                  ->orWhere('machine', 'like', "%{$search}%");
            });
        }

        $productions = $query->latest('date')->latest('id')->paginate(20)->withQueryString();

        // Summary
        $totals = [
            'raw_material' => Production::sum('raw_material_cost'),
            'total_cost' => Production::sum('total_cost'),
            'output_qty' => Production::sum('output_quantity'),
            'count' => Production::count(),
        ];

        return view('production.index', compact('productions', 'totals'));
    }

    /**
     * Show the form for creating a new production.
     */
    public function create(): View
    {
        $rawItems = Item::where('product_type', 'raw')->where('is_active', true)->orderBy('name')->get();
        $finishItems = Item::where('product_type', 'finish')->where('is_active', true)->orderBy('name')->get();
        $nextBatchNo = Production::generateBatchNo();

        return view('production.create', compact('rawItems', 'finishItems', 'nextBatchNo'));
    }

    /**
     * Store a newly created production.
     */
    public function store(Request $request): RedirectResponse
    {
        $request->validate([
            'date' => 'required|date',
            'machine' => 'nullable|string|max:100',
            'input_items' => 'required|array|min:1',
            'input_items.*.item_id' => 'required|exists:items,id',
            'input_items.*.quantity' => 'required|numeric|min:0.001',
            'output_items' => 'required|array|min:1',
            'output_items.*.item_id' => 'required|exists:items,id',
            'output_items.*.quantity' => 'required|numeric|min:0.001',
            'agent_commission' => 'nullable|numeric|min:0',
            'transportation_cost' => 'nullable|numeric|min:0',
            'labor_cost' => 'nullable|numeric|min:0',
            'electricity_cost' => 'nullable|numeric|min:0',
            'other_expenses' => 'nullable|numeric|min:0',
        ]);

        try {
            DB::beginTransaction();

            // Calculate raw material cost
            $rawMaterialCost = 0;
            foreach ($request->input_items as $item) {
                if (!empty($item['item_id']) && !empty($item['quantity'])) {
                    $itemModel = Item::find($item['item_id']);
                    $rawMaterialCost += $item['quantity'] * $itemModel->purchase_price;
                }
            }

            // Calculate output quantity
            $outputQuantity = 0;
            foreach ($request->output_items as $item) {
                if (!empty($item['item_id']) && !empty($item['quantity'])) {
                    $outputQuantity += $item['quantity'];
                }
            }

            // Calculate total cost
            $totalCost = $rawMaterialCost
                + ($request->agent_commission ?? 0)
                + ($request->transportation_cost ?? 0)
                + ($request->labor_cost ?? 0)
                + ($request->electricity_cost ?? 0)
                + ($request->other_expenses ?? 0);

            // Calculate cost per unit
            $costPerUnit = $outputQuantity > 0 ? $totalCost / $outputQuantity : 0;

            // Create production
            $production = Production::create([
                'batch_no' => Production::generateBatchNo(),
                'date' => $request->date,
                'machine' => $request->machine,
                'raw_material_cost' => $rawMaterialCost,
                'agent_commission' => $request->agent_commission ?? 0,
                'transportation_cost' => $request->transportation_cost ?? 0,
                'labor_cost' => $request->labor_cost ?? 0,
                'electricity_cost' => $request->electricity_cost ?? 0,
                'other_expenses' => $request->other_expenses ?? 0,
                'other_expenses_description' => $request->other_expenses_description,
                'total_cost' => $totalCost,
                'output_quantity' => $outputQuantity,
                'wastage_quantity' => $request->wastage_quantity ?? 0,
                'cost_per_unit' => $costPerUnit,
                'narration' => $request->narration,
                'created_by' => Auth::id(),
            ]);

            // Create input items (raw materials) and reduce stock
            foreach ($request->input_items as $item) {
                if (!empty($item['item_id']) && !empty($item['quantity'])) {
                    $itemModel = Item::find($item['item_id']);
                    $amount = $item['quantity'] * $itemModel->purchase_price;

                    ProductionItem::create([
                        'production_id' => $production->id,
                        'item_id' => $item['item_id'],
                        'type' => 'input',
                        'quantity' => $item['quantity'],
                        'rate' => $itemModel->purchase_price,
                        'amount' => $amount,
                    ]);

                    // Reduce raw material stock
                    $itemModel->reduceStock($item['quantity'], 'production', "Production Batch: {$production->batch_no}");
                }
            }

            // Create output items (finished goods) and add stock
            foreach ($request->output_items as $item) {
                if (!empty($item['item_id']) && !empty($item['quantity'])) {
                    $itemModel = Item::find($item['item_id']);

                    ProductionItem::create([
                        'production_id' => $production->id,
                        'item_id' => $item['item_id'],
                        'type' => 'output',
                        'quantity' => $item['quantity'],
                        'rate' => $costPerUnit,
                        'amount' => $item['quantity'] * $costPerUnit,
                    ]);

                    // Add finished goods stock
                    $itemModel->addStock($item['quantity'], 'production', "Production Batch: {$production->batch_no}");

                    // Update item's cost price to calculated cost per unit
                    $itemModel->update(['purchase_price' => $costPerUnit]);
                }
            }

            DB::commit();

            return redirect()->route('production.show', $production)
                ->with('success', "Production batch {$production->batch_no} created. Cost per unit: Rs. " . number_format($costPerUnit, 2));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withInput()->with('error', 'Error creating production: ' . $e->getMessage());
        }
    }

    /**
     * Display the specified production.
     */
    public function show(Production $production): View
    {
        $production->load(['inputItems.item', 'outputItems.item', 'creator']);
        return view('production.show', compact('production'));
    }

    /**
     * Show the form for editing the specified production.
     */
    public function edit(Production $production): View
    {
        $production->load(['inputItems.item', 'outputItems.item']);
        $rawItems = Item::where('product_type', 'raw')->where('is_active', true)->orderBy('name')->get();
        $finishItems = Item::where('product_type', 'finish')->where('is_active', true)->orderBy('name')->get();

        return view('production.edit', compact('production', 'rawItems', 'finishItems'));
    }

    /**
     * Update the specified production.
     */
    public function update(Request $request, Production $production): RedirectResponse
    {
        $request->validate([
            'date' => 'required|date',
            'machine' => 'nullable|string|max:100',
            'agent_commission' => 'nullable|numeric|min:0',
            'transportation_cost' => 'nullable|numeric|min:0',
            'labor_cost' => 'nullable|numeric|min:0',
            'electricity_cost' => 'nullable|numeric|min:0',
            'other_expenses' => 'nullable|numeric|min:0',
        ]);

        try {
            DB::beginTransaction();

            // Recalculate total cost with new expenses
            $totalCost = $production->raw_material_cost
                + ($request->agent_commission ?? 0)
                + ($request->transportation_cost ?? 0)
                + ($request->labor_cost ?? 0)
                + ($request->electricity_cost ?? 0)
                + ($request->other_expenses ?? 0);

            // Recalculate cost per unit
            $costPerUnit = $production->output_quantity > 0 ? $totalCost / $production->output_quantity : 0;

            // Update production
            $production->update([
                'date' => $request->date,
                'machine' => $request->machine,
                'agent_commission' => $request->agent_commission ?? 0,
                'transportation_cost' => $request->transportation_cost ?? 0,
                'labor_cost' => $request->labor_cost ?? 0,
                'electricity_cost' => $request->electricity_cost ?? 0,
                'other_expenses' => $request->other_expenses ?? 0,
                'other_expenses_description' => $request->other_expenses_description,
                'total_cost' => $totalCost,
                'cost_per_unit' => $costPerUnit,
                'narration' => $request->narration,
            ]);

            // Update output items rate and amount
            foreach ($production->outputItems as $outputItem) {
                $outputItem->update([
                    'rate' => $costPerUnit,
                    'amount' => $outputItem->quantity * $costPerUnit,
                ]);

                // Update item's cost price
                $outputItem->item->update(['purchase_price' => $costPerUnit]);
            }

            DB::commit();

            return redirect()->route('production.show', $production)
                ->with('success', "Production updated. New cost per unit: Rs. " . number_format($costPerUnit, 2));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withInput()->with('error', 'Error updating production: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified production.
     */
    public function destroy(Production $production): RedirectResponse
    {
        try {
            DB::beginTransaction();

            // Reverse stock changes
            // Add back raw materials
            foreach ($production->inputItems as $item) {
                $item->item->addStock($item->quantity, 'production_delete', "Deleted Production: {$production->batch_no}");
            }

            // Reduce finished goods
            foreach ($production->outputItems as $item) {
                $item->item->reduceStock($item->quantity, 'production_delete', "Deleted Production: {$production->batch_no}");
            }

            $batchNo = $production->batch_no;
            $production->items()->delete();
            $production->delete();

            DB::commit();

            return redirect()->route('production.index')
                ->with('success', "Production batch {$batchNo} deleted and stock reversed.");
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Error deleting production: ' . $e->getMessage());
        }
    }

    /**
     * Print production batch
     */
    public function print(Production $production): View
    {
        $production->load(['inputItems.item', 'outputItems.item', 'creator']);
        return view('production.print', compact('production'));
    }
}
